<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    public function up(): void
    {
        // ----------------------
        // Products / Items
        // ----------------------
        Schema::create('manufacturing_products', function (Blueprint $table) {
            $table->id();
            $table->string('sku')->unique();
            $table->string('name');
            $table->text('description')->nullable();
            $table->string('unit', 20)->default('pcs');
            $table->decimal('price', 15,2)->default(0);
            $table->timestampsTz();
        });

        // ----------------------
        // Suppliers
        // ----------------------
        Schema::create('manufacturing_suppliers', function (Blueprint $table) {
            $table->id();
            $table->string('name');
            $table->string('contact_person')->nullable();
            $table->string('email')->nullable();
            $table->string('phone')->nullable();
            $table->string('address')->nullable();
            $table->timestampsTz();
        });

        // ----------------------
        // Purchase Orders
        // ----------------------
        Schema::create('manufacturing_purchase_orders', function (Blueprint $table) {
            $table->id();
            $table->unsignedBigInteger('supplier_id'); // removed FK
            $table->date('order_date');
            $table->enum('status', ['pending','received','cancelled'])->default('pending');
            $table->timestampsTz();
        });

        // Purchase Order Items
        Schema::create('manufacturing_purchase_order_items', function (Blueprint $table) {
            $table->id();
            $table->unsignedBigInteger('purchase_order_id'); // removed FK
            $table->unsignedBigInteger('product_id'); // removed FK
            $table->decimal('quantity',15,2);
            $table->decimal('price',15,2)->default(0);
            $table->timestampsTz();
        });

        // ----------------------
        // Bill of Materials (BOM)
        // ----------------------
        Schema::create('manufacturing_bom', function (Blueprint $table) {
            $table->id();
            $table->unsignedBigInteger('product_id'); // removed FK
            $table->unsignedBigInteger('raw_material_id'); // removed FK
            $table->decimal('quantity',15,2);
            $table->timestampsTz();
        });

        // ----------------------
        // Work Orders / Production Orders
        // ----------------------
        Schema::create('manufacturing_work_orders', function (Blueprint $table) {
            $table->id();
            $table->unsignedBigInteger('product_id'); // removed FK
            $table->integer('quantity')->default(0);
            $table->integer('produced_quantity')->default(0);
            $table->integer('rejected_quantity')->default(0);
            $table->enum('status',['pending','in_progress','completed','cancelled'])->default('pending');
            $table->date('start_date')->nullable();
            $table->date('end_date')->nullable();
            $table->timestampsTz();
        });

        // ----------------------
        // Stock Movements
        // ----------------------
        Schema::create('manufacturing_stock_movements', function (Blueprint $table) {
            $table->id();
            $table->unsignedBigInteger('product_id'); // removed FK
            $table->enum('movement_type',['raw_material','finished_goods','scrap'])->default('raw_material');
            $table->decimal('quantity',15,2);
            $table->string('batch_no')->nullable();
            $table->text('remarks')->nullable();
            $table->timestampsTz();
        });

        // ----------------------
        // Machines / Resources
        // ----------------------
        Schema::create('manufacturing_machines', function (Blueprint $table) {
            $table->id();
            $table->string('name');
            $table->text('description')->nullable();
            $table->enum('status',['active','inactive','maintenance'])->default('active');
            $table->timestampsTz();
        });

        // ----------------------
        // Production Operations / Routing
        // ----------------------
        Schema::create('manufacturing_operations', function (Blueprint $table) {
            $table->id();
            $table->unsignedBigInteger('work_order_id'); // removed FK
            $table->string('operation_name');
            $table->unsignedBigInteger('machine_id')->nullable(); // removed FK
            $table->unsignedBigInteger('assigned_to')->nullable(); // removed FK
            $table->decimal('estimated_hours',5,2)->default(0);
            $table->decimal('actual_hours',5,2)->nullable();
            $table->enum('status',['pending','in_progress','completed'])->default('pending');
            $table->timestampsTz();
        });

        // ----------------------
        // Quality Checks
        // ----------------------
        Schema::create('manufacturing_quality_checks', function (Blueprint $table) {
            $table->id();
            $table->unsignedBigInteger('work_order_id'); // removed FK
            $table->enum('status',['pass','fail']);
            $table->text('remarks')->nullable();
            $table->timestampsTz();
        });

        // ----------------------
        // Machine Maintenance / Downtime
        // ----------------------
        Schema::create('manufacturing_machine_maintenance', function (Blueprint $table) {
            $table->id();
            $table->unsignedBigInteger('machine_id'); // removed FK
            $table->date('maintenance_date');
            $table->enum('status',['scheduled','completed','missed'])->default('scheduled');
            $table->text('remarks')->nullable();
            $table->timestampsTz();
        });
    }

    public function down(): void
    {
        Schema::dropIfExists('manufacturing_machine_maintenance');
        Schema::dropIfExists('manufacturing_quality_checks');
        Schema::dropIfExists('manufacturing_operations');
        Schema::dropIfExists('manufacturing_machines');
        Schema::dropIfExists('manufacturing_stock_movements');
        Schema::dropIfExists('manufacturing_work_orders');
        Schema::dropIfExists('manufacturing_bom');
        Schema::dropIfExists('manufacturing_purchase_order_items');
        Schema::dropIfExists('manufacturing_purchase_orders');
        Schema::dropIfExists('manufacturing_suppliers');
        Schema::dropIfExists('manufacturing_products');
    }
};
