<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    public function up(): void
    {
        // ----------------------
        // Currencies Table
        // ----------------------
        Schema::create('currencies', function (Blueprint $table) {
            $table->id();
            $table->string('name'); // e.g., US Dollar
            $table->string('code', 10)->unique(); // e.g., USD
            $table->string('symbol', 5); // e.g., $
            $table->unsignedTinyInteger('decimal_places')->default(2); // e.g., 2 decimal points
            $table->boolean('is_active')->default(true); // active or not
            $table->timestampsTz();
            $table->softDeletesTz();
        });

        // ----------------------
        // Exchange Rates Table
        // ----------------------
        Schema::create('currency_exchange_rates', function (Blueprint $table) {
            $table->id();
            $table->foreignId('base_currency_id')->constrained('currencies')->cascadeOnDelete(); // e.g., USD
            $table->foreignId('target_currency_id')->constrained('currencies')->cascadeOnDelete(); // e.g., EUR
            $table->decimal('rate', 18, 6); // e.g., 0.923456
            $table->date('date')->default(DB::raw('CURRENT_DATE')); // rate effective date
            $table->timestampsTz();

            $table->unique(['base_currency_id','target_currency_id','date'], 'currency_rate_unique');
        });

        // ----------------------
        // Optional: Company-specific default currency
        // ----------------------
        Schema::create('company_currencies', function (Blueprint $table) {
            $table->id();
            $table->unsignedBigInteger('company_id'); // references companies table (optional FK)
            $table->foreignId('currency_id')->constrained('currencies')->cascadeOnDelete();
            $table->boolean('is_default')->default(false); // default currency for company
            $table->timestampsTz();
            $table->unique(['company_id','currency_id']);
        });

        // ----------------------
        // Optional: User-specific currency preference
        // ----------------------
        Schema::create('user_currencies', function (Blueprint $table) {
            $table->id();
            $table->foreignId('user_id')->constrained('users')->cascadeOnDelete();
            $table->foreignId('currency_id')->constrained('currencies')->cascadeOnDelete();
            $table->timestampsTz();
            $table->unique(['user_id','currency_id']);
        });
    }

    public function down(): void
    {
        Schema::dropIfExists('user_currencies');
        Schema::dropIfExists('company_currencies');
        Schema::dropIfExists('currency_exchange_rates');
        Schema::dropIfExists('currencies');
    }
};
