<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration {

    public function up(): void
    {
        /*------------------------------------------
         | 1️⃣ Categories (Multi-Level)
         ------------------------------------------*/
        Schema::create('categories', function (Blueprint $table) {
            $table->id();
            $table->uuid('uuid')->unique();

            // Removed FK
            $table->unsignedBigInteger('company_id')->nullable();
            $table->unsignedBigInteger('parent_id')->nullable();

            $table->string('name');
            $table->timestampsTz();
        });

        /*------------------------------------------
         | 2️⃣ Brands
         ------------------------------------------*/
        Schema::create('brands', function (Blueprint $table) {
            $table->id();
            $table->uuid('uuid')->unique();

            // Removed FK
            $table->unsignedBigInteger('company_id')->nullable();

            $table->string('name');
            $table->timestampsTz();
        });

        /*------------------------------------------
         | 3️⃣ Units (PCS, KG, LTR, BOX...)
         ------------------------------------------*/
        Schema::create('units', function (Blueprint $table) {
            $table->id();
            $table->uuid('uuid')->unique();

            // Removed FK
            $table->unsignedBigInteger('company_id')->nullable();

            $table->string('name');
            $table->string('short_code');
            $table->timestampsTz();
        });

        /*------------------------------------------
         | 4️⃣ Products (Main Table)
         ------------------------------------------*/
        Schema::create('products', function (Blueprint $table) {
            $table->id();
            $table->uuid('uuid')->unique();

            // Removed FK
            $table->unsignedBigInteger('company_id')->nullable();
            $table->unsignedBigInteger('branch_id')->nullable();

            $table->string('name');
            $table->string('sku')->unique();
            $table->enum('type', ['stockable','service','digital'])->default('stockable');

            $table->unsignedBigInteger('category_id')->nullable();
            $table->unsignedBigInteger('brand_id')->nullable();
            $table->unsignedBigInteger('unit_id')->nullable();

            $table->boolean('is_serialized')->default(false);
            $table->boolean('is_expirable')->default(false);

            $table->decimal('cost_price', 15, 2)->default(0);
            $table->decimal('sale_price', 15, 2)->default(0);

            $table->jsonb('attributes')->nullable();

            $table->timestampsTz();
            $table->softDeletesTz();
        });

        /*------------------------------------------
         | 5️⃣ Product Attributes (Custom Fields)
         ------------------------------------------*/
        Schema::create('product_attributes', function (Blueprint $table) {
            $table->id();
            $table->uuid('uuid')->unique();

            // Removed FK
            $table->unsignedBigInteger('company_id')->nullable();

            $table->string('name');
            $table->enum('type', ['string','number','select'])->default('string');
            $table->boolean('is_required')->default(false);

            $table->timestampsTz();
        });

        /*------------------------------------------
         | 6️⃣ Product Attribute Values
         ------------------------------------------*/
        Schema::create('product_attribute_values', function (Blueprint $table) {
            $table->id();
            $table->uuid('uuid')->unique();

            // Removed FK
            $table->unsignedBigInteger('product_id')->nullable();
            $table->unsignedBigInteger('attribute_id')->nullable();

            $table->string('value');
            $table->timestampsTz();
        });

        /*------------------------------------------
         | 7️⃣ Product Serials / IMEI / Unique Keys
         ------------------------------------------*/
        Schema::create('product_serials', function (Blueprint $table) {
            $table->id();
            $table->uuid('uuid')->unique();

            // Removed FK
            $table->unsignedBigInteger('product_id')->nullable();
            $table->unsignedBigInteger('company_id')->nullable();

            $table->string('serial_number')->unique();
            $table->boolean('is_sold')->default(false);
            $table->date('warranty_until')->nullable();

            $table->timestampsTz();
        });

        /*------------------------------------------
         | 8️⃣ Inventory Stock
         ------------------------------------------*/
        Schema::create('inventory_stock', function (Blueprint $table) {
            $table->id();
            $table->uuid('uuid')->unique();

            // Removed FK
            $table->unsignedBigInteger('product_id')->nullable();
            $table->unsignedBigInteger('company_id')->nullable();
            $table->unsignedBigInteger('branch_id')->nullable();

            $table->decimal('quantity', 15, 2)->default(0);
            $table->date('expiry_date')->nullable();

            $table->timestampsTz();
        });

        /*------------------------------------------
         | 9️⃣ Stock Movements (In/Out)
         ------------------------------------------*/
        Schema::create('stock_movements', function (Blueprint $table) {
            $table->id();
            $table->uuid('uuid')->unique();

            // Removed FK
            $table->unsignedBigInteger('product_id')->nullable();
            $table->unsignedBigInteger('company_id')->nullable();
            $table->unsignedBigInteger('branch_id')->nullable();

            $table->enum('type', ['in','out','adjustment']);
            $table->decimal('quantity', 15, 2);
            $table->decimal('cost_price', 15, 2)->nullable();
            $table->jsonb('meta')->nullable();

            $table->timestampsTz();
        });
    }

    public function down(): void
    {
        Schema::dropIfExists('stock_movements');
        Schema::dropIfExists('inventory_stock');
        Schema::dropIfExists('product_serials');
        Schema::dropIfExists('product_attribute_values');
        Schema::dropIfExists('product_attributes');
        Schema::dropIfExists('products');
        Schema::dropIfExists('units');
        Schema::dropIfExists('brands');
        Schema::dropIfExists('categories');
    }
};
